package gov.va.vamf.scheduling.varutility.resources;

import gov.va.vamf.scheduling.varutility.domain.CustomMessage;
import gov.va.vamf.scheduling.varutility.domain.CustomMessages;
import gov.va.vamf.scheduling.varutility.clientapi.MobileHealthClientTestVersion;
import gov.va.vamf.scheduling.varutility.testutility.TestHelper;
import gov.va.vamf.scheduling.varutility.testutility.IntegrationTestConfiguration;

import org.junit.*;

import javax.ws.rs.client.Client;
import javax.ws.rs.client.Entity;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.UriBuilder;

import java.net.URI;
import java.util.Iterator;

public class VarUtilityCustomMessageTest {

    private static MobileHealthClientTestVersion client;
    private static Client jerseyClient;
    private static String[] siteCodes = {"test-site-code-1", "test-site-code-2"};
    private static String MESSAGE_TEXT_1 = "Save message text one.";
    private static String MESSAGE_TEXT_2 = "Replaced message text one with message text two.";
    private static int MESSAGE_COUNT_ONE = 1;
    private static String[] messageIds = {"appointmentNoPreferredDatePopup", "typeOfCareDropdown", "noClinicRadioButtons"};

    @BeforeClass
    public static void startup() {
        client = TestHelper.createProviderClient();
        jerseyClient = client.getJerseyClient();
    }

    @AfterClass
    public static void destroy() {
        if (client != null) {
            client.getJerseyClient().close();
            client = null;
        }
    }

    @Test
    public void testFillCustomMessagesWithDefaultMessages() {
        testFillDefaultMessages();
        testDeleteCustomMessages();
    }

    @Test
    public void testFetchDefaultMessages() {
        for (int i = 0; i < messageIds.length; i++) {
            testFetchDefaultMessage(messageIds[i]);
        }
    }

    @Test
    public void testSaveAndReplace() {
        testFillDefaultMessages();
        testSaveCustomMessages();
        testReplaceCustomMessages();
        testDeleteCustomMessages();
    }

    @Test
    public void testPartialFillDefaultMessageAfterSave() {
        testSaveCustomMessage(siteCodes[0], messageIds[0], MESSAGE_TEXT_1);
        testSaveCustomMessage(siteCodes[1], messageIds[1], MESSAGE_TEXT_1);
        testFillDefaultMessages();
        testDeleteCustomMessages();
    }

    public void testFetchDefaultMessage(String messageId) {
        CustomMessage findDefaultMessage = jerseyClient.target(getDefaultMessageResourceUri(messageId))
                .request()
                .accept(MediaType.APPLICATION_JSON)
                .get(CustomMessage.class);

        Assert.assertNotNull(findDefaultMessage);
        Assert.assertEquals(findDefaultMessage.getMessageId(), messageId);
    }

    private void testFillDefaultMessages() {
        for (int i = 0; i < siteCodes.length; i++) {
            CustomMessages findCustomMessages = jerseyClient.target(getCustomMessageResourceUri(siteCodes[i]))
                    .request()
                    .accept(MediaType.APPLICATION_JSON)
                    .get(CustomMessages.class);

            Assert.assertNotNull(findCustomMessages);
            Assert.assertEquals(findCustomMessages.size(), messageIds.length);

            for (int j = 0; j < messageIds.length; j++) {
                Assert.assertTrue(containsMessageId(messageIds[j], findCustomMessages));
            }
        }
    }

    private void testSaveCustomMessages() {

        for (int i = 0; i < siteCodes.length; i++) {
            for (int j = 0; j < messageIds.length; j++) {
                testSaveCustomMessage(siteCodes[i], messageIds[j], MESSAGE_TEXT_1);
            }
        }
    }

    private void testReplaceCustomMessages() {

        for (int i = 0; i < siteCodes.length; i++) {
            for (int j = 0; j < messageIds.length; j++) {
                testSaveCustomMessage(siteCodes[i], messageIds[j], MESSAGE_TEXT_2);
            }
        }
    }

    private void testSaveCustomMessage(String siteCode, String messageId, String messageText) {
        CustomMessages putPayload = createCustomMessagesPayload(messageId, messageText);

        CustomMessages savedCustomMessages = jerseyClient.target(getCustomMessageResourceUri(siteCode))
                .request(MediaType.APPLICATION_JSON)
                .accept(MediaType.APPLICATION_JSON)
                .put(Entity.json(putPayload), CustomMessages.class);

        Assert.assertNotNull(savedCustomMessages);
        Assert.assertEquals(savedCustomMessages.size(), MESSAGE_COUNT_ONE);
        Assert.assertTrue(containsMessageId(messageId, savedCustomMessages));
    }

    private void testDeleteCustomMessages() {
        for (int i = 0; i < siteCodes.length; i++) {
            CustomMessages deletedCustomMessages = jerseyClient.target(getCustomMessageResourceUri(siteCodes[i]))
                    .request()
                    .accept(MediaType.APPLICATION_JSON)
                    .delete(CustomMessages.class);

            Assert.assertNotNull(deletedCustomMessages);
        }
    }

    private CustomMessages createCustomMessagesPayload(String messageId, String textMessage) {
        CustomMessages newCustomMessages = new CustomMessages();

        CustomMessage customMessage = new CustomMessage();
        customMessage.setMessageId(messageId);
        customMessage.setMessageText(textMessage);

        newCustomMessages.add(customMessage);

        return newCustomMessages;
    }

    private static URI getCustomMessageResourceUri(String siteCode) {
        String uriFormat = String.format("%s/custom-messages/site/%s",
                IntegrationTestConfiguration.getServerBaseUri(), siteCode);

        return UriBuilder.fromUri(uriFormat).build();
    }

    private static URI getDefaultMessageResourceUri(String messageId) {
        String uriFormat = String.format("%s/custom-messages/default-message/message-id/%s",
                IntegrationTestConfiguration.getServerBaseUri(), messageId);

        return UriBuilder.fromUri(uriFormat).build();
    }

    private boolean containsMessageId(String messageId, CustomMessages customMessages) {
        Iterator<CustomMessage> customMessagesIterator = customMessages.iterator();

        while (customMessagesIterator.hasNext()) {
            if (messageId.equals(customMessagesIterator.next().getMessageId())) {
                return true;
            }
        }

        return false;
    }
}
